/*++

Copyright (c) 2011 Vasily Tarasov. All rights reserved.

Module Name:

    main.c

Abstract:

    Command-line utility for arranging multiple monitors.

    Source code is provided free of charge. It is free to be used for
    reference purposes, as a code base or as it is in both commercial
    and non-commercial projects.

Author:

    Vasily Tarasov (vtarasov@zoneos.com)    Jan 25, 2011

Revision History:

    Jan 25, 2011 Vasily Tarasov (vtarasov@zoneos.com)
      - Created.

    Feb 1, 2011 Vasily Tarasov (vtarasov@zoneos.com)
      - Option --setmode added to set width, height and color depth.

    Dec 19, 2011 Vasily Tarasov (vtarasov@zoneos.com)
      - Options --list that print available display devices.

--*/


#include <windows.h>
#include <tchar.h>


VOID
PrintHeader(
)
/*++

Routine Description:

    Prints copyright notice at application startup.

Arguments:

    VOID

Return Value:

    VOID

--*/
{
    wprintf(
        L"Copyright (c) 2011 Vasily Tarasov. All rights reserved.\n"
        L"Multiple monitors rearrangement tool.\n"
        L"\n"
    );
}


VOID
PrintHelp(
)
/*++

Routine Description:

    Prints help page for the application.

Arguments:

    VOID

Return Value:

    VOID

--*/
{
    wprintf(
        L"Usage:\n"
        L"    monarr --setmode <display> <width> <height> <bpp>\n"
        L"    monarr --move <display> <x> <y>\n"
        L"\n"
        L"    !!! WARNING !!!\n"
        L"    There is no 15-seconds rollback option.\n"
        L"    Don't try to set new mode unless you are absolutely sure\n"
        L"    it is supported by the hardware and your monitor.\n"
        L"\n"
        L"  --setmode\n"
        L"    Tool sets width, height and color depth for monitor\n"
        L"    <display> to specified values.\n"
        L"\n"
        L"  --move\n"
        L"    Tool sets upper-left corner coordinates for monitor\n"
        L"    <display> to values (x,y). Usually, primary monitor have\n"
        L"    these coordinates set to (0,0).\n"
        L"\n"
        L"  --list [description substring]\n"
        L"    Print list of display devices with description matching\n"
        L"    given substring. Search is case-sensitive.\n"
        L"\n"
        L"  --listall - print all display devices in the system\n"
        L"\n"
        L"  --listvirtual - same as --list ZoneScreen\n"
        L"\n"
        L"  --help - print this page\n"
        L"\n"
        L"Examples:\n"
        L"    monarr --setmode \\\\.\\DISPLAY1 1024 768 32\n"
        L"    This command sets mode 1024x768 at 32bpp for first display.\n"
        L"\n"
        L"    monarr --move \\\\.\\DISPLAY2 1024 0\n"
        L"    This will move second display to the right of primary\n"
        L"    display. Primary monitor here is 1024 pixels in width.\n"
        L"\n"
        L"    monarr --list ZoneScreen\n"
        L"\n"
    );
}


PCWSTR
ModeChangeStatusDescription(
    IN LONG Status
)
/*++

Routine Description:

    Describes value, returned by ChangeDisplaySettingsEx() function.
    Description strings are taken from official MSDN documentation
    except for default case for unknown value.

Arguments:

    Status - value to get description for.

Return Value:

    Unicode string that describes status code.

--*/
{
    PCWSTR Meaning;


    switch( Status ) {
        case DISP_CHANGE_SUCCESSFUL:
            Meaning = L"The settings change was successful.";
            break;

        case DISP_CHANGE_BADDUALVIEW:
            Meaning = L"The settings change was unsuccessful because "
                L"the system is DualView capable.";
            break;

        case DISP_CHANGE_BADFLAGS:
            Meaning = L"An invalid set of flags was passed in.";
            break;

        case DISP_CHANGE_BADMODE:
            Meaning = L"The graphics mode is not supported.";
            break;

        case DISP_CHANGE_BADPARAM:
            Meaning = L"An invalid parameter was passed in. This can "
                L"include an invalid flag or combination of flags.";
            break;

        case DISP_CHANGE_FAILED:
            Meaning = L"The display driver failed the specified graphics "
                L"mode.";
            break;

        case DISP_CHANGE_NOTUPDATED:
            Meaning = L"Unable to write settings to the registry.";
            break;

        case DISP_CHANGE_RESTART:
            Meaning = L"The computer must be restarted for the graphics "
                L"mode to work.";
            break;

        default:
            // This string is the only one that is not from MSDN.
            Meaning = L"Operation failed for unknown reason.";
            break;
    }

    return Meaning;
}


LONG
ZsSetDisplayMode(
    IN PCWSTR DeviceName,
    IN ULONG Width,
    IN ULONG Height,
    IN ULONG ColorDepth
)
/*++

Routine Description:

    Sets specified mode (width x height x color depth) for specified display.
    Display name is usually given in the form '\\.\DISPLAY5',
    where 5 is display index used as an example.

Arguments:

    DeviceName - display name to set mode for.
    Width - target width to be set.
    Height - target heidht to be set.
    ColorDepth - target color depth, e.g. 16, 24 or 32.

Return Value:

    Status value, see MSDN and ModeChangeStatusDescription for detailed
    list of possible values.

--*/
{
    DEVMODEW DevMode;


    memset( &DevMode, 0, sizeof(DevMode) );
    DevMode.dmSize = sizeof(DevMode);

    DevMode.dmFields =
        DM_BITSPERPEL |
        DM_PELSWIDTH |
        DM_PELSHEIGHT;

    DevMode.dmPelsWidth = Width;
    DevMode.dmPelsHeight = Height;
    DevMode.dmBitsPerPel = ColorDepth;

    return ChangeDisplaySettingsEx(
        DeviceName,
        &DevMode,
        NULL,
        CDS_UPDATEREGISTRY,
        NULL
    );
}


LONG
ZsMoveDisplay(
    IN PCWSTR DeviceName,
    IN PPOINTL Position
)
/*++

Routine Description:

    Sets position for display in a multi-monitor environment. Display name
    is usually given in the form '\\.\DISPLAY5', where 5 is display index
    used as an example.

Arguments:

    DeviceName
        Display name to set position for.

    Position
        Pointer to POINTL structure that describes upper-left position
        to set for specified display.

Return Value:

    Status value, see MSDN and ModeChangeStatusDescription for detailed
    list of possible values.

--*/
{
    DEVMODEW DevMode;


    memset( &DevMode, 0, sizeof(DevMode) );
    DevMode.dmSize = sizeof(DevMode);

    DevMode.dmFields = DM_POSITION;
    DevMode.dmPosition = *Position;

    return ChangeDisplaySettingsEx(
        DeviceName,
        &DevMode,
        NULL,
        CDS_UPDATEREGISTRY,
        NULL
    );
}


typedef VOID (*ZS_ENUM_CALLBACK)(PDISPLAY_DEVICEW DisplayInfo);


VOID
ZsEnumDispayDevices(
    IN ZS_ENUM_CALLBACK EnumCallback,
    IN PCWSTR DescriptionSubstring OPTIONAL
)
/*++

Routine Description:

    Enumerate all display devices in the system, and pass display device
    information to a given callback.

Arguments:

    EnumCallback
        Callback to pass information about every appropriate device
        for handling.

    DescriptionSubstring
        If not NULL, only devices mathing this description will be
        passed to callback.

Return Value:

    VOID

--*/
{
    DISPLAY_DEVICEW DisplayInfo;
    ULONG DeviceNumber = 0;
    BOOL QueryStatus;


    for( DeviceNumber = 0; ; ++DeviceNumber ) {

        DisplayInfo.cb = sizeof( DisplayInfo );

        QueryStatus = EnumDisplayDevices(
            NULL,           // Device Name
            DeviceNumber,   // Device Number
            &DisplayInfo,   // Device Information
            0               // Flags
        );

        // No more display devices
        if( !QueryStatus ) {
            break;
        }

        // Skip mirroring drivers
        if( DisplayInfo.StateFlags & DISPLAY_DEVICE_MIRRORING_DRIVER ) {
            continue;
        }

        if( DescriptionSubstring != NULL ) {
            if( wcsstr(
                DisplayInfo.DeviceString,
                DescriptionSubstring
            ) == NULL )
            {
                // Skip if description was provided but it doesn't match
                continue;
            }
        }

        EnumCallback( &DisplayInfo );
    }
}


VOID
ZsEnumCallback(
    IN PDISPLAY_DEVICEW DisplayInfo
)
/*++

Routine Description:

    Prints appropriate information about given display device.

Arguments:

    DisplayInfo
        Complete display information structure obtained during
        display enumeration.

Return Value:

    VOID

--*/
{
    wprintf( L"%s\n", DisplayInfo->DeviceName );
}


LONG
__cdecl
wmain(
    IN LONG argc,
    IN PWSTR argv[]
)
/*++

Routine Description:

    Entry point for the application.

Arguments:

    argc - number of command line arguments.
    argv - array of command line arguments.

Return Value:

    0 on success
    1 on failure

--*/
{
    PCWSTR DeviceName;
    POINTL Position;
    PCWSTR Meaning;
    ULONG Width;
    ULONG Height;
    ULONG ColorDepth;
    LONG Result;


    PrintHeader();

    if( 0 == wcscmp( argv[1], L"--help" ) || 0 == wcscmp( argv[1], L"-h" ) ||
        0 == wcscmp( argv[1], L"/?" ) || 0 == wcscmp( argv[1], L"-?" ) )
    {
        PrintHelp();
        return 0;
    }

    if( 0 == wcscmp( argv[1], L"--setmode" ) ) {

        if( argc < 6 ) {
            wprintf( L"Mandatory options are missing. "
                L"See --help for usage.\n" );
            return 1;
        }

        DeviceName = argv[2];
        Width = _wtoi( argv[3] );
        Height = _wtoi( argv[4] );
        ColorDepth = _wtoi( argv[5] );

        Result = ZsSetDisplayMode( DeviceName, Width, Height, ColorDepth );
        Meaning = ModeChangeStatusDescription( Result );
        wprintf( L"%s\n", Meaning );

    } else if( 0 == wcscmp( argv[1], L"--move" ) ) {

        if( argc < 5 ) {
            wprintf( L"Mandatory options are missing. "
                L"See --help for usage.\n" );
            return 1;
        }

        DeviceName = argv[2];
        Position.x = _wtoi( argv[3] );
        Position.y = _wtoi( argv[4] );

        Result = ZsMoveDisplay( DeviceName, &Position );
        Meaning = ModeChangeStatusDescription( Result );
        wprintf( L"%s\n", Meaning );

    } else if( 0 == wcscmp( argv[1], L"--list" ) ) {
        DeviceName = argc > 2 ? argv[2] : NULL;
        ZsEnumDispayDevices( &ZsEnumCallback, DeviceName );
    } else if( 0 == wcscmp( argv[1], L"--listvirtual" ) ) {
        ZsEnumDispayDevices( &ZsEnumCallback, L"ZoneScreen" );
    } else if( 0 == wcscmp( argv[1], L"--listall" ) ) {
        ZsEnumDispayDevices( &ZsEnumCallback, NULL );

    } else {
        wprintf( L"Unknown option: '%s'. See --help for usage.\n", argv[1] );
        return 1;
    }

    return 0;
}
