//  Copyright (C) 2005-2006 Lev Kazarkin. All Rights Reserved.
//
//  TightVNC is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
//  USA.
//
// TightVNC distribution homepage on the Web: http://www.tightvnc.com/

#include "VideoDriver.h"
#include "vncDesktop.h"

char	vncVideoDriver::szDriverString[] = "ZoneScreen";

BOOL IsWinNT();
BOOL IsNtVer(ULONG mj, ULONG mn);
BOOL IsWinVerOrHigher(ULONG mj, ULONG mn);


vncVideoDriver::vncVideoDriver()
{
    bufdata.Updates = NULL;
    bufdata.FrameBufferView = NULL;
	m_fIsActive = false;
	m_fDirectAccessInEffect = false;
	m_fHandleScreen2ScreenBlt = false;
	*m_devname= 0;
}

vncVideoDriver::~vncVideoDriver()
{
	UnMapSharedbuffers();
	Deactivate();
	_ASSERTE(!m_fIsActive);
	_ASSERTE(!m_fDirectAccessInEffect);
}

#define	BYTE0(x)	((x) & 0xFF)
#define	BYTE1(x)	(((x) >> 8) & 0xFF)
#define	BYTE2(x)	(((x) >> 16) & 0xFF)
#define	BYTE3(x)	(((x) >> 24) & 0xFF)

BOOL vncVideoDriver::CheckVersion()
{
	_ASSERTE(IsWinNT());

	HDC	l_gdc= ::CreateDC(m_devname, NULL, NULL, NULL);
	if (!l_gdc)
	{
		vnclog.Print(
			LL_INTERR,
			VNCLOG("vncVideoDriver::CheckVersion: can't create DC on \"%s\"\n"),
			m_devname);
		return FALSE;
	}

	vnclog.Print(
		LL_INTINFO,
		"Supported driver version is: %u\n",
		ZS_CURRENT_UPDATES_ENGINE_VERSION);

	ULONG driverVersion;

	int drvCr = ExtEscape(
		l_gdc,
        ZS_ESC_QUERY_VERSION,
		0, NULL,
		sizeof(driverVersion), (LPSTR) &driverVersion);
	DeleteDC(l_gdc);

	if (drvCr == 0)
	{
		vnclog.Print(
			LL_INTERR,
			VNCLOG("vncVideoDriver::CheckVersion: ZS_ESC_QUERY_VERSION not supported by this version of driver\n"));
		return FALSE;
	}

	vnclog.Print(
		LL_INTINFO,
		"Driver version is: %u\n",
		driverVersion);

	if (drvCr < 0)
	{
		vnclog.Print(
			LL_INTERR,
			VNCLOG("vncVideoDriver::CheckVersion: ZS_ESC_QUERY_VERSION call returned 0x%x\n"),
			drvCr);
		return FALSE;
	}

	return TRUE;
}

BOOL vncVideoDriver::MapSharedbuffers(BOOL fForDirectScreenAccess)
{
	_ASSERTE(!m_fIsActive);
	_ASSERTE(!m_fDirectAccessInEffect);
	_ASSERTE(IsWinNT());

	HDC	l_gdc= ::CreateDC(m_devname, NULL, NULL, NULL);
	if (!l_gdc)
	{
		vnclog.Print(
			LL_INTERR,
			VNCLOG("vncVideoDriver::MapSharedbuffers: can't create DC on \"%s\"\n"),
			m_devname);
		return FALSE;
	}

	oldCounter = 0;
	int drvCr = ExtEscape(
		l_gdc,
        ZS_ESC_MAP_BUFFERS,
		0, NULL,
		sizeof(bufdata), (LPSTR) &bufdata);
	DeleteDC(l_gdc);

	if (drvCr <= 0)
	{
		vnclog.Print(
			LL_INTERR,
			VNCLOG("vncVideoDriver::MapSharedbuffers: MAP1 call returned 0x%x\n"),
			drvCr);
		return FALSE;
	}

	m_fIsActive = true;

    // If direct access was requested by UI and is supported by driver.
    m_fDirectAccessInEffect = fForDirectScreenAccess && bufdata.FrameBufferView != NULL;

	m_fHandleScreen2ScreenBlt = true;

	return TRUE;	
}

void vncVideoDriver::UnMapSharedbuffers()
{
	_ASSERTE(IsWinNT());

	vnclog.Print(
		LL_INTINFO,
		VNCLOG("vncVideoDriver::UnMapSharedbuffers failed. unmapping manually\n"));
	if (bufdata.Updates)
	{
        BOOL br = UnmapViewOfFile(bufdata.Updates);
		vnclog.Print(
			LL_INTINFO,
			VNCLOG("vncVideoDriver::UnMapSharedbuffers: UnmapViewOfFile(bufdata.buffer) returned %d\n"),
			br);
	}
    if (bufdata.FrameBufferView)
	{
		BOOL br = UnmapViewOfFile(bufdata.FrameBufferView);
		vnclog.Print(
			LL_INTINFO,
			VNCLOG("vncVideoDriver::UnMapSharedbuffers: UnmapViewOfFile(bufdata.Userbuffer) returned %d\n"),
			br);
	}

	m_fIsActive = false;
	m_fDirectAccessInEffect = false;
	m_fHandleScreen2ScreenBlt = false;
}

template <class TpFn>
HINSTANCE  LoadNImport(LPCTSTR szDllName, LPCTSTR szFName, TpFn &pfn)
{
	HINSTANCE hDll = LoadLibrary(szDllName);
	if (hDll)
	{
		pfn = (TpFn)GetProcAddress(hDll, szFName);
		if (pfn)
			return hDll;
		FreeLibrary(hDll);
	}
	vnclog.Print(
		LL_INTERR,
		VNCLOG("Can not import '%s' from '%s'.\n"),
		szFName, szDllName);
	return NULL;
}

BOOL vncVideoDriver::LookupVideoDevice(LPCTSTR szDeviceString, INT &devNum, DISPLAY_DEVICE *pDd)
{
	_ASSERTE(IsWinVerOrHigher(5, 0));

	pEnumDisplayDevices pd = NULL;
	HINSTANCE  hInstUser32 = LoadNImport("User32.DLL", "EnumDisplayDevicesA", pd);
	if (!hInstUser32) return FALSE;

	ZeroMemory(pDd, sizeof(DISPLAY_DEVICE));
	pDd->cb = sizeof(DISPLAY_DEVICE);
	BOOL result;
	while (result = (*pd)(NULL,devNum, pDd, 0))
	{
		if (NULL != strstr((const char *)pDd->DeviceString, szDeviceString))
		{
			vnclog.Print(
				LL_INTINFO,
				VNCLOG("Found display device \"%s\": \"%s\"\n"),
				pDd->DeviceString,
				pDd->DeviceName);
			break;
		}
		devNum++;
	}

	FreeLibrary(hInstUser32);
	return result;
}

BOOL CALLBACK vncVideoDriver::monitorEnumProc(
    HMONITOR hMonitor,
    HDC hdcMonitor,
    LPRECT lprcMonitor,
    LPARAM dwData
)
{
    MONITORINFOEXA monitorInfo;
    vncVideoDriver* _this = (vncVideoDriver*)dwData;

    monitorInfo.cbSize = sizeof(monitorInfo);
    ::GetMonitorInfoA( hMonitor, &monitorInfo );

    if( !strcmp( _this->GetDeviceName(), monitorInfo.szDevice ) )
    {
        _this->SetCurrentRect( *lprcMonitor );
        return FALSE;
    }

    return TRUE;
}

BOOL vncVideoDriver::Activate(
		BOOL fForDirectAccess,
		const RECT *prcltarget)
{
	_ASSERTE(IsWinNT());

	DISPLAY_DEVICE dd;
	INT devNum = 0;
	if (!LookupVideoDevice(szDriverString, devNum, &dd))
	{
		vnclog.Print(LL_INTERR, VNCLOG("No '%s' found.\n"), szDriverString);
		return FALSE;
	}

    strcpy(m_devname, (const char *)dd.DeviceName);

    // To get current display rectangle.
    EnumDisplayMonitors(
        NULL,
        NULL,
        &monitorEnumProc,
        (LPARAM)this
    );

	return IsWinVerOrHigher(5, 0);
}

void vncVideoDriver::Deactivate()
{
	_ASSERTE(IsWinNT());
}

void vncVideoDriver::HandleDriverChanges(
		vncDesktop *pDesk,
		vncRegion &rgn,
		int xoffset,
		int yoffset,
		BOOL &bPointerShapeChange)
{
    ULONG snapshot_counter = bufdata.Updates->CurrentPosition;
	if (oldCounter == snapshot_counter)
		return;

	if (oldCounter < snapshot_counter)
	{
		HandleDriverChangesSeries(
			pDesk,
			rgn,
			xoffset, yoffset,
			bufdata.Updates->Updates + oldCounter,
			bufdata.Updates->Updates + snapshot_counter,
			bPointerShapeChange);
	}
	else
	{
		HandleDriverChangesSeries(
			pDesk,
			rgn,
			xoffset, yoffset,
            bufdata.Updates->Updates + oldCounter,
			bufdata.Updates->Updates + ZS_UPDATES_BUFFER_LENGTH,
			bPointerShapeChange);

		HandleDriverChangesSeries(
			pDesk,
			rgn,
			xoffset, yoffset,
			bufdata.Updates->Updates,
			bufdata.Updates->Updates + snapshot_counter,
			bPointerShapeChange);
	}

	oldCounter = snapshot_counter;
}

void vncVideoDriver::HandleDriverChangesSeries(
		vncDesktop *pDesk,
		vncRegion &rgn,
		int xoffset,
		int yoffset,
		const ZS_UPDATE_RECORD *i,
		const ZS_UPDATE_RECORD *last,
		BOOL &bPointerShapeChange)
{
	for (; i < last; i++)
	{
// TODO bPointerShapeChange

		if (m_fHandleScreen2ScreenBlt && i->Type == ZsUpdateMove)
		{
		//	DPF(("CopyRect: (%d, %d, %d, %d)\n",
		//		i->rect.left,
		//		i->rect.top,
		//		i->rect.right,
		//		i->rect.bottom));

			RECT Rc;
			Rc.left = i->Rectangle.left + xoffset;
			Rc.top = i->Rectangle.top + yoffset;
			Rc.right = i->Rectangle.right + xoffset;
			Rc.bottom = i->Rectangle.bottom + yoffset;

			POINT Pt;
			Pt.x = i->Point.x + xoffset;
			Pt.y = i->Point.y + yoffset;
			pDesk->CopyRect(Rc, Pt);

            continue;
		}

		if (i->Type == ZsUpdateOther)
		{
		//	DPF(("XRect: (%d, %d, %d, %d)\n",
		//		i->rect.left,
		//		i->rect.top,
		//		i->rect.right,
		//		i->rect.bottom));
			rgn.AddRect(i->Rectangle, xoffset, yoffset);
		}
	}
}
